package edu.cmu.cs.cs214.blockingqueue;


/**
 * This class implements a SynchronousQueue.  
 *
 * Like any basic queue, the SynchronousQueue has enqueue and dequeue 
 * methods.  The SynchronousQueue, though, does not store any elements.
 * Instead, any enqueue (or dequeue) call waits for a corresponding
 * dequeue (or enqueue) call, at which time the enqueued element is
 * passed to the dequeue method.
 * 
 * Note that enqueue and dequeue on a SynchronousQueue will always eventually
 * succeed, as long as another thread eventually dequeues or enqueues, 
 * respectively.

 * @param <E> - The type of the elements in the BlockingQueue
 */
public class SynchronousQueue<E> implements BlockingQueue<E> {
    E item;
    boolean readyToDequeue;
    final Object lock;
    
    /** 
     * Constructs a SynchronousQueue.
     */
    public SynchronousQueue() {
    	lock = new Object();
    	item = null;
    	readyToDequeue = false;
    }

    /**
     * Waits for a corresponding dequeue call, to synchronously pass the enqueued
     * element to the dequeue call.
     * 
     * @param e  The element to be passed to the corresponding dequeue call.
     */
    public void enqueue(E e) {
        synchronized(lock) {
            while (readyToDequeue) {  // Must wait if we're already storing an item
                try {
                    lock.wait();  // Blocks here until interrupted or notified.
                } catch (InterruptedException ignore) { /* ignore */ }
            }
            item = e;
            readyToDequeue = true;
            lock.notifyAll();
        }
    }
    
    /**
     * Removes and returns the element from the front of the queue if the queue
     * is not empty, or blocks (and eventually removes and returns the front element) 
     * if the queue is empty.
     * 
     * @return  The item removed from the front of the queue.
     */
    public E dequeue() {
    	synchronized(lock) {
            while (!readyToDequeue) {  // Must wait if we're not storing an item
                try {
                    lock.wait();       // Blocks here until interrupted or notified.
                } catch (InterruptedException ignore) { /* ignore */ }
            }
            E rv = item;
            readyToDequeue = false;
            lock.notifyAll();
            return rv;
        }
    }
    
    public static void main(String[] args) {
    	// Starts new threads for a producer and a consumer to access a 
    	// single queue.
        SynchronousQueue<Integer> queue = new SynchronousQueue<Integer>();
        Producer p = new Producer(queue);
        Producer p2 = new Producer(queue);
        Consumer c = new Consumer(queue);
        new Thread(p).start();
        new Thread(p2).start();
        new Thread(c).start();
    }
}

